/*********************************************************************************************
*	Copyright (C) 2002 Robert Farrell
*
*	This program is free software; you can redistribute it and/or
*	modify it under the terms of the GNU General Public License
*	as published by the Free Software Foundation; either version 2
*	of the License, or (at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
*
*	See the GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
**********************************************************************************************/
/*********************************************************************************************
*	Class:	cMultiProperty
*
*	Purpose:	
*
*	This class encapsulates a single property with a parameter name, value, default
*	value, and informational comment that can have zero or more additional values
*	associated with it.
*				
*	Notes:	
*
*	An example of a multiple value property would be:
*	server1 = http://www.kickbutt.com; Fragfest 24/7; 34567; 20; true
*	This might represent the parameter name of server1 with five values
*	for server_address, server_name, server_port, connect_rety_count,
*	and ping_on_startup. This way, five values can be associated with the parameter
*	name: server1.
*
*	Each of the multiple property values has a value and a default value.
*	For a cMultiProperty with a single value, it's represented by index 0.
*	The multi-values start at index 1. This class guarantees that the total number 
*	of values and default values are always equal.
*
*	This class offers no type conversion safety by itself. Type changes occur when
*	any Set* () member functions are called. A call to a Set* () member function will
*	cause a corresponding type change for either the default or regular value depending
*	if Set* () or Set*Default () was called.
*	For instance, a call to SetBool () will cause a change to the value's type but
*	will not change the default value's type to cPropertyValue::PV_BOOL.
*
*	New values are created in the Set* () and Type () member functions if an index
*	is sent that is out of range. The new value will be added to next available index.
*	For instance, if there are 3 values, and a call to SetBool ( true, 3 ) is made,
*	a new value will be created at m_values [ 3 ]. If a call to SetBool ( true, 10 )
*	is made and there are only 3 values, the new value is still only added to index 3.
*
*	When a Get* () function is called with an index that is out of range, the highest
*	valid index value is returned.
*
*	The types for values and default values are not guaranteed to be the same for
*	identical indexes. So if Type ( 1 ) is set to PV_INT, TypeDefault ( 1 ) is not
*	changed and may not be the same (and vice versa). Internally this shouldn't matter
*	much since type conversions can occur for any value.
*
*	Author:	Robert Farrell
*	Email:	farrellr@accis.edu
*
*	History:
*	12/18/2001	Initial version.
*********************************************************************************************/
#ifndef _MULTI_PROPERTY_HPP
#define _MULTI_PROPERTY_HPP

#include "ps_value.hpp"
#include <vector>

using std::string;

class cMultiProperty
{
public:
	cMultiProperty ( const cPropertyValue::eValueType type );
	cMultiProperty ();
	virtual ~cMultiProperty ();

	cPropertyValue::eValueType	Type (	const unsigned short index = 0 );
	void						Type (	const cPropertyValue::eValueType type, 
										const unsigned short index = 0 );

	cPropertyValue::eValueType	TypeDefault (	const unsigned short index = 0 );
	void						TypeDefault (	const cPropertyValue::eValueType type, 
												const unsigned short index = 0 );
	string	Name () const;
	void	Name ( const string name );
	string	Info () const;
	void	Info ( const string info );
	
	// values
	bool	GetBool ( const unsigned short index = 0 );
	void	SetBool ( const bool value, const unsigned short index = 0 );
	char	GetChar ( const unsigned short index = 0 );
	void	SetChar ( const char value, const unsigned short index = 0 );
	short	GetShort ( const unsigned short index = 0 );
	void	SetShort ( const short value, const unsigned short index = 0 );
	int		GetInt ( const unsigned short index = 0 );
	void	SetInt ( const int value, const unsigned short index = 0 );
	long	GetLong ( const unsigned short index = 0 );
	void	SetLong ( const long value, const unsigned short index = 0 );
	float	GetFloat ( const unsigned short index = 0 );
	void	SetFloat ( const float value, const unsigned short index = 0 );
	double	GetDouble ( const unsigned short index = 0 );
	void	SetDouble ( const double value, const unsigned short index = 0 );
	string	GetString ( const unsigned short index = 0 );
	void	SetString ( const string value, const unsigned short index = 0 );

	// default values
	bool	GetBoolDefault ( const unsigned short index = 0 );
	void	SetBoolDefault ( const bool value, const unsigned short index = 0 );
	char	GetCharDefault ( const unsigned short index = 0 );
	void	SetCharDefault ( const char value, const unsigned short index = 0 );
	short	GetShortDefault ( const unsigned short index = 0 );
	void	SetShortDefault ( const short value, const unsigned short index = 0 );
	int		GetIntDefault ( const unsigned short index = 0 );
	void	SetIntDefault ( const int value, const unsigned short index = 0 );
	long	GetLongDefault ( const unsigned short index = 0 );
	void	SetLongDefault ( const long value, const unsigned short index = 0 );
	float	GetFloatDefault ( const unsigned short index = 0 );
	void	SetFloatDefault ( const float value, const unsigned short index = 0 );
	double	GetDoubleDefault ( const unsigned short index = 0 );
	void	SetDoubleDefault ( const double value, const unsigned short index = 0 );
	string	GetStringDefault ( const unsigned short index = 0 );
	void	SetStringDefault ( const string value, const unsigned short index = 0 );

	// sets all property values to cPropertValue::PV_NONE, does not deallocate memory
	void	Clear ();

	// deallocates memory for all values, allocates values for index 0
	void	Destroy ();
	bool	IsValidIndex ( const unsigned short index ) const;
	
	unsigned int NumValues () const;

private:
	bool	IsValidDefaultIndex ( const unsigned short index ) const;
	void	AllocateValue ( const cPropertyValue::eValueType type = cPropertyValue::PV_NONE );

private:
	string							m_name;
	string							m_info;
	std::vector < cPropertyValue >	m_values;
	std::vector < cPropertyValue >	m_defaults;
};
#endif // _MULTI_PROPERTY_HPP
