/*********************************************************************************************
*	Copyright (C) 2002 Robert Farrell
*
*	This program is free software; you can redistribute it and/or
*	modify it under the terms of the GNU General Public License
*	as published by the Free Software Foundation; either version 2
*	of the License, or (at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
*
*	See the GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
**********************************************************************************************/
#include "ps_property_set.hpp"
#include "ps_reader.hpp"
#include "ps_writer.hpp"
#include <utility>

using std::make_pair;
using std::ios;

cPropertySet::cPropertySet ( const string &filename, 
							 const cPropertySetConfig::eFileMode mode ) 
{
	config.SetFileName ( filename );
	config.SetFileMode ( mode );

	cPropertySetReader reader ( *this );
	reader.Read ();
}

cPropertySet::~cPropertySet ()
{
	cPropertySetWriter writer ( *this );
	writer.Write ();
	
	Destroy ();	
}

void cPropertySet::Clear ()
{
	std::multimap < string, cMultiProperty >::iterator key;
	
	for ( key = m_property_set.begin (); key != m_property_set.end (); ++key )
		key->second.Clear ();
}

void cPropertySet::Destroy ()
{
	std::multimap < string, cMultiProperty >::iterator key;
	
	for ( key = m_property_set.begin (); key != m_property_set.end (); ++key )
		key->second.Destroy ();
}

unsigned short cPropertySet::NumValues ( const string &key, const string &name )
{
	if ( !IsParamExist ( key, name ) )
		return 0;

	else
	{
		cMultiProperty * p = RetrieveMultiProperty ( key, name );
		return p->NumValues ();
	}
}

bool cPropertySet::IsValidValueIndex (	const string &key, 
										const string &name, 
										const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return false;

	else
	{
		cMultiProperty * p = RetrieveMultiProperty ( key, name );
		return p->IsValidIndex ( index );
	}
}

void
cPropertySet::SetBool ( const string &key,
						const string &name, 
						const bool value, 
						const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetBool ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetBool ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void 
cPropertySet::SetChar ( const string &key, 
						const string &name, 
						const char value, 
						const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetChar ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetChar ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void 
cPropertySet::SetShort ( const string &key, 
						 const string &name,
						 const short value,
						 const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetShort ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetShort ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void 
cPropertySet::SetInt (	const string &key, 
						const string &name, 
						const int value,
						const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetInt ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetInt ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void 
cPropertySet::SetLong ( const string &key, 
						const string &name, 
						const long value,
						const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetLong ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetLong ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void 
cPropertySet::SetFloat ( const string &key, 
						 const string &name, 
						 const float value,
						 const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetFloat ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetFloat ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void 
cPropertySet::SetDouble ( const string &key, 
						  const string &name, 
						  const double value,
						  const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetDouble ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetDouble ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void 
cPropertySet::SetString ( const string &key, 
						  const string &name, 
						  const string &value,
						  const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetString ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetString ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void 
cPropertySet::SetInfo (	const string &key, 
						const string &name, 
						const string &info )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->Info ( info );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.Info ( info );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void
cPropertySet::SetBoolDefault (	const string &key, 
								const string &name, 
								const bool value,
								const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetBoolDefault ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetBoolDefault ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}


void
cPropertySet::SetCharDefault (	const string &key, 
								const string &name, 
								const char value,
								const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetCharDefault ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetCharDefault ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void
cPropertySet::SetShortDefault (	const string &key, 
								const string &name, 
								const short value,
								const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetShortDefault ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetShortDefault ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void
cPropertySet::SetIntDefault (	const string &key, 
								const string &name, 
								const int value,
								const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetIntDefault ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetIntDefault ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void
cPropertySet::SetLongDefault (	const string &key, 
								const string &name, 
								const long value,
								const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetLongDefault ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetLongDefault ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void
cPropertySet::SetFloatDefault (	const string &key, 
								const string &name, 
								const float value,
								const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetFloatDefault ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetFloatDefault ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void
cPropertySet::SetDoubleDefault ( const string &key, 
								 const string &name, 
								 const double value,
								 const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetDoubleDefault ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetDoubleDefault ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

void
cPropertySet::SetStringDefault ( const string &key, 
								 const string &name, 
								 const string &value,
								 const unsigned short index )
{
	// if the param exists, replace the value
	if ( IsParamExist ( key, name ) )
	{
		cMultiProperty * cur = 0;
		cur = RetrieveMultiProperty ( key, name );
		cur->SetStringDefault ( value, index );
	}
	else
	{
		// no need to replace, just add it
		cMultiProperty cur;
		cur.Name ( name );
		cur.SetStringDefault ( value );
		m_property_set.insert( make_pair ( key, cur ) );
	}
}

bool
cPropertySet::GetBool ( const string &key,
						const string &name,
						const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return false;

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetBool ( index );
}

char
cPropertySet::GetChar ( const string &key, 
						const string &name,
						const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return '\0';

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetChar ( index );
}

short
cPropertySet::GetShort ( const string &key,
						 const string &name,
						 const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return 0;

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetShort ( index );
}

int
cPropertySet::GetInt ( const string &key,
					   const string &name,
					   const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return 0;

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetInt ( index );
}

long
cPropertySet::GetLong ( const string &key,
						const string &name,
						const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return 0;

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetLong ( index );
}

float
cPropertySet::GetFloat ( const string &key,
						 const string &name,
						 const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return 0.0;

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetFloat ( index );
}

double 
cPropertySet::GetDouble ( const string &key, 
						  const string &name,
						  const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return 0.0;

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetDouble ( index );
}

string
cPropertySet::GetString (	const string &key,
							const string &name,
							const unsigned short index )
{
	// return empty string if key/param doesn't exist
	if ( !IsParamExist ( key, name ) )
		return "";

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetString ( index );
}

bool
cPropertySet::GetBoolDefault ( const string &key,
							   const string &name,
							   const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return false;

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetBoolDefault ( index );
}

char
cPropertySet::GetCharDefault ( const string &key,
							   const string &name,
							   const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return '\0';

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetCharDefault ( index );
}

short
cPropertySet::GetShortDefault (  const string &key,
								 const string &name,
								 const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return 0;

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetShortDefault ( index );
}

int
cPropertySet::GetIntDefault ( const string &key, 
							  const string &name,
							  const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return 0;

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetIntDefault ( index );
}

long
cPropertySet::GetLongDefault ( const string &key, 
							   const string &name,
							   const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return 0;

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetLongDefault ( index );
}

float
cPropertySet::GetFloatDefault ( const string &key, 
							    const string &name,
							    const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return 0.0;

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetFloatDefault ( index );
}

double
cPropertySet::GetDoubleDefault ( const string &key, 
								 const string &name,
								 const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return 0.0;

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetFloatDefault ( index );
}

string
cPropertySet::GetStringDefault ( const string &key, 
								 const string &name,
								 const unsigned short index )
{
	if ( !IsParamExist ( key, name ) )
		return "";

	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->GetStringDefault ( index );
}

string
cPropertySet::GetInfo ( const string &key, 
						const string &name )
{
	// return empty string if key/param doesn't exist
	if ( !IsParamExist ( key, name ) )
		return "";
	
	cMultiProperty * cur = 0;

	cur = RetrieveMultiProperty ( key, name );
	return cur->Info ();
}

bool	cPropertySet::IsKeyExist ( const string &key ) const
{
	if ( m_property_set.find ( key ) != m_property_set.end () )
		return true;

	return false;
}

bool	cPropertySet::IsParamExist ( const string &key, const string &param )
{
	if ( !IsKeyExist ( key ) )
		return false;

	std::multimap < string, cMultiProperty >::const_iterator cur_param;

	// try to find the parameter name using for the sent key
	for ( cur_param = m_property_set.lower_bound ( key ); cur_param != m_property_set.upper_bound ( key ); ++cur_param )
	{
		if ( cur_param->second.Name () == param )
			return true;
	}

	return false;
}

cMultiProperty * cPropertySet::RetrieveMultiProperty ( const string &key, const string &param )
{
	std::multimap < string, cMultiProperty >::iterator cur_param;

	// try to find the parameter name
	for ( cur_param = m_property_set.lower_bound ( key ); cur_param != m_property_set.upper_bound ( key ); ++cur_param )
	{
		if ( cur_param->second.Name () == param )
			return &(cur_param->second);
	}

	// not there
	return 0;
}

cPropertySet::cFileModeConstraint::cFileModeConstraint ( string msg ) :
cPropertyException ( msg )
{
}


