/*********************************************************************************************
*	Copyright (C) 2002 Robert Farrell
*
*	This program is free software; you can redistribute it and/or
*	modify it under the terms of the GNU General Public License
*	as published by the Free Software Foundation; either version 2
*	of the License, or (at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
*
*	See the GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
**********************************************************************************************/
/*********************************************************************************************
*	Class:		cPropertySet
*
*	Purpose:	To store name/value pairs and name/multi-value pairs sorted by keys.
*				
*	Notes:	
*
*	Keys:	Delimiters are not stored in the key, there is no validation.
*	Parameters:
*	Values: 
*	Default Values:
*	Info: Start delimiters are not stored in the info string, there is no validation.
*
*	Property set file is read upon construction and saved upon destruction.
*
*	Rules for missing keys and/or parameters:
*	GetBool* () returns false if key or parameter name is not found
*	GetChar* () returns '\0' if key or parameter name is not found
*	GetShort* () returns 0 if key or parameter name is not found
*	GetInt* () returns 0 if key or parameter name is not found
*	GetLong* () returns 0 if key or parameter name is not found
*	GetFloat* () returns 0.0 if key or parameter name is not found
*	GetDouble* () returns 0.0 if key or parameter name is not found
*	GetString* () returns "" if key or parameter name is not found
*
*	FIXME:	1. cPropertySetConfig is read during construction and changes are 
*			only reflected in cPropertySet destruction where the config info 
*			is re-read (pointless to change it unless you want to change the file
*			the stuff is written to basically). This is as it should be.
*			2. Add logging for retrieval of missing keys and/or params.
*			3. Info associated with a key is stored as key=keyname, param=keyname,
*			there may be a better way to do this.
*			4. Add a footer variable to store info not associated with any keys
*			or params that should come at the end of a file.
*
*	Author:	Robert Farrell
*	Email:	farrellr@accis.edu
*
*	History:
*	12/18/2001	Initial version.
*	1/8/2002	Key-start, key-end and info-start delimiters are not stored in the class.
*	1/11/2002	IsParamExist and RetrieveMultiProperty now only search the sent key's
*				parameters rather than all keys and parameters. This caused the first matching
*				parameter name to be retrieved regardless of the key it was found in. 
*********************************************************************************************/
#ifndef _PROPERTY_SET_HPP
#define _PROPERTY_SET_HPP

#pragma warning( disable:4786 )

#include "ps_multi_property.hpp"
#include "ps_config.hpp"
#include <map>
#include <fstream>

using std::fstream;
using std::string;
using std::multimap;

class cPropertySet
{
	friend class cPropertySetWriter;

public:
	cPropertySetConfig config;

	// Exceptions
	class cFileModeConstraint;

	cPropertySet (	const string &filename, 
					const cPropertySetConfig::eFileMode mode = cPropertySetConfig::EXISTS_OVERWRITE_OR_CREATE );

	virtual ~cPropertySet ();

	// calls Clear () for each multi value
	void	Clear ();

	// calls Destroy () for each multi value
	void	Destroy ();
	bool	IsValidValueIndex ( const string &key, const string &name, const unsigned short index );

	unsigned short NumValues ( const string &key, const string &name );

	void	SetBool ( const string &key, const string &name, const bool value, const unsigned short index = 0 );
	void	SetChar ( const string &key, const string &name, const char value, const unsigned short index = 0 );
	void	SetShort ( const string &key, const string &name, const short value, const unsigned short index = 0 );
	void	SetInt ( const string &key, const string &name, const int value, const unsigned short index = 0 );
	void	SetLong ( const string &key, const string &name, const long value, const unsigned short index = 0 );
	void	SetFloat ( const string &key, const string &name, const float value, const unsigned short index = 0 );
	void	SetDouble ( const string &key, const string &name, const double value, const unsigned short index = 0 );
	void	SetString ( const string &key, const string &name, const string &value, const unsigned short index = 0 );
	void	SetInfo ( const string &key, const string &name, const string &info );
	
	void	SetBoolDefault ( const string &key, const string &name, const bool value, const unsigned short index = 0 );
	void	SetCharDefault ( const string &key, const string &name, const char value, const unsigned short index = 0 );
	void	SetShortDefault ( const string &key, const string &name, const short value, const unsigned short index = 0 );
	void	SetIntDefault ( const string &key, const string &name, const int value, const unsigned short index = 0 );
	void	SetLongDefault ( const string &key, const string &name, const long value, const unsigned short index = 0 );
	void	SetFloatDefault ( const string &key, const string &name, const float value, const unsigned short index = 0 );
	void	SetDoubleDefault ( const string &key, const string &name, const double value, const unsigned short index = 0 );
	void	SetStringDefault ( const string &key, const string &name, const string &value, const unsigned short index = 0 );
	
	bool	GetBool ( const string &key, const string &name, const unsigned short index = 0 );
	char	GetChar ( const string &key, const string &name, const unsigned short index = 0 );
	short	GetShort ( const string &key, const string &name, const unsigned short index = 0 );
	int		GetInt ( const string &key, const string &name, const unsigned short index = 0 );
	long	GetLong ( const string &key, const string &name, const unsigned short index = 0 );
	float	GetFloat ( const string &key, const string &name, const unsigned short index = 0 );
	double	GetDouble ( const string &key, const string &name, const unsigned short index = 0 );
	string	GetString ( const string &key, const string &name, const unsigned short index = 0 );
	string	GetInfo ( const string &key, const string &name );

	bool	GetBoolDefault ( const string &key, const string &name, const unsigned short index = 0 );
	char	GetCharDefault ( const string &key, const string &name, const unsigned short index = 0 );
	short	GetShortDefault ( const string &key, const string &name, const unsigned short index = 0 );
	int		GetIntDefault ( const string &key, const string &name, const unsigned short index = 0 );
	long	GetLongDefault ( const string &key, const string &name, const unsigned short index = 0 );
	float	GetFloatDefault ( const string &key, const string &name, const unsigned short index = 0 );
	double	GetDoubleDefault ( const string &key, const string &name, const unsigned short index = 0 );
	string	GetStringDefault ( const string &key, const string &name, const unsigned short index = 0 );
	
	bool	IsKeyExist ( const string &key ) const;
	bool	IsParamExist ( const string &key, const string &param );

private:
	// returns 0 if not found
	cMultiProperty * RetrieveMultiProperty ( const string &key, const string &param ); 

private:
	std::multimap < string, cMultiProperty >	m_property_set;
};

class cPropertySet::cFileModeConstraint : public cPropertyException
{
public:
	cFileModeConstraint ( string msg );
};
#endif // _PROPERTY_SET_HPP
