/*********************************************************************************************
*	Copyright (C) 2002 Robert Farrell
*
*	This program is free software; you can redistribute it and/or
*	modify it under the terms of the GNU General Public License
*	as published by the Free Software Foundation; either version 2
*	of the License, or (at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
*
*	See the GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
**********************************************************************************************/
#include "ps_reader.hpp"
#include <sstream>


#ifdef _DEBUG
#include <iostream>
using std::cout;
using std::endl;
#endif // _DEBUG

using std::stringstream;
using std::ios;

cPropertySetReader::cPropertySetReader ( cPropertySet &ps ) :
m_property_set ( &ps ),
m_cur_state ( PARAM_START ),
m_prev_state ( PARAM_START ),
M_INFO_START ( ps.config.GetInfoStartDelim () ),
M_KEY_START ( ps.config.GetKeyStartDelim () ),
M_KEY_END ( ps.config.GetKeyEndDelim () ),
M_LINE_END ( ps.config.GetLineEndDelim () ),
M_ASSIGN ( ps.config.GetAssignSymbol () ),
M_MULTI_VALUE ( ps.config.GetMultiValueDelim () )
{
	// if parameters are found before the first key, they are assigned to this key
	m_cur_key = "NONE";
	m_in.unsetf ( ios::skipws );

	stringstream buffer;
	string file = ps.config.GetFileName ();

	// open the file based on the constraints
	if ( ps.config.GetFileMode () == cPropertySetConfig::EXISTS_OVERWRITE_OR_CREATE )
	{
		// attempt to open existing file
		m_in.open ( file.c_str (), ios::in );
		if ( !m_in )
		{
			// file doesn't exist so create it
			m_in.clear ();
			m_in.open ( file.c_str (), ios::out );
			if ( !m_in )
			{
				buffer << "Error: File IO error while attempting to create file: " << file.c_str ();
				cPropertySet::cFileModeConstraint e ( buffer.str () );
				throw e;
			}
		}
	}
	else
	if ( ps.config.GetFileMode () == cPropertySetConfig::EXISTS_OVERWRITE_OR_FAIL )
	{
		// attempt to open existing file
		m_in.open ( file.c_str (), ios::in );
		if ( !m_in )
		{
			// file doesn't exist so fail
			buffer << "Error: File IO error while attempting to open existing file: " << file.c_str ();
			cPropertySet::cFileModeConstraint e ( buffer.str () );
			throw e;
		}
		else
		{
			// file exists, so overwrite
			m_in.close ();
			m_in.clear ();
			m_in.open ( file.c_str (), ios::out );
			if ( !m_in )
			{
				// file io error
				buffer << "Error: File IO error while attempting to overwrite existing file: " << file.c_str ();
				cPropertySet::cFileModeConstraint e ( buffer.str () );
				throw e;
			}
		}
	}
	else
	if ( ps.config.GetFileMode () == cPropertySetConfig::EXISTS_FAIL_OR_CREATE )
	{
		// see if the file exists
		m_in.open ( file.c_str (), ios::in );
		if ( !m_in )
		{
			// file doesn't exist so create it
			m_in.clear ();
			m_in.open ( file.c_str (), ios::out );
			if ( !m_in )
			{
				buffer << "Error: File IO error while attempting to create file: " << file.c_str ();
				cPropertySet::cFileModeConstraint e ( buffer.str () );
				throw e;
			}
		}
		else
		{
			// file exists, so fail
			buffer << "Error: File: " << file.c_str () << " exists and should not";
			cPropertySet::cFileModeConstraint e ( buffer.str () );
			throw e;
		}
	}
	else
	{
		buffer << "Error: File constraint not specified for file: " << file.c_str ();
		cPropertySet::cFileModeConstraint e ( buffer.str () );
		throw e;
	}
}
cPropertySetReader::~cPropertySetReader ()
{
	m_in.clear ();
	m_in.close ();
}

void cPropertySetReader::Read ()
{
	// parse the file char by char until EOF is returned
	char token = EOF;
	while ( (token = GetNextChar ()) != EOF )
	{
		switch ( m_cur_state )
		{
			case INFO:			ProcessInfo ( token );			break;
			case PARAM_START:	ProcessParamStart ( token );	break;
			case KEY:			ProcessKey ( token );			break;
			case KEY_END:		ProcessKeyEnd ( token );		break;
			case PARAM:			ProcessParam ( token );			break;
			case VALUE:			ProcessValue ( token );			break;
			case MULTI_VALUE:	ProcessMultiValue ( token );	break;

			default: break;
		}

		if ( m_cur_state == PARSE_ERROR )
			ProcessError ();
	}

	ProcessEOF ();
}

char cPropertySetReader::GetNextChar ()
{
	char next = EOF;

	m_in >> next;
	return next;
}

void cPropertySetReader::ProcessInfo ( const char token )
{
	if ( m_cur_state != INFO || !m_property_set )
		return;
	
	m_cur_info += token;

	if ( token == M_LINE_END )
	{
		TrimWhiteSpace ( m_cur_info );
		m_cur_multi_info.push_back ( m_cur_info );
		m_cur_info = "";
		m_cur_state = PARAM_START;
	}
}

void cPropertySetReader::ProcessParamStart ( const char token )
{
	if ( m_cur_state != PARAM_START || !m_property_set )
		return;

	if ( token == M_INFO_START )
	{	
		// strip the delimiter by not accumulating it
		m_cur_state = INFO;
	}
	else
	if ( token == M_KEY_START )
	{
		// strip the key delim by discarding the token
		m_cur_key = "";
		m_cur_state = KEY;
	}
	else
	if ( token == M_KEY_END )
	{
		m_prev_state = m_cur_state;
		m_cur_state = PARSE_ERROR;
	}
	else
	if ( token == M_LINE_END )
	{
		// do nothing
	}
	else
	if ( token == M_ASSIGN )
	{
		m_prev_state = m_cur_state;
		m_cur_state = PARSE_ERROR;
	}
	else
	if ( token == ' ' || token == '\t' )
	{
		// ignore white space
	}
	else
	if ( token == M_MULTI_VALUE )
	{
		m_prev_state = m_cur_state;
		m_cur_state = PARSE_ERROR;
	}
	else
	{
		m_cur_param = token;
		m_cur_state = PARAM;
	}
}

void cPropertySetReader::ProcessKey ( const char token )
{
	if ( m_cur_state != KEY || !m_property_set )
		return;

	if ( token == M_KEY_END )
	{
		// discard the key-end delimiter token
		TrimWhiteSpace ( m_cur_key );
		m_cur_state = KEY_END;
	}
	else
	{
		m_cur_key += token;
	}
}

void cPropertySetReader::ProcessKeyEnd ( const char token )
{
	if ( m_cur_state != KEY_END || !m_property_set )
		return;
	
#ifdef _DEBUG
	cout << "have end of key..." << endl;				
	DisplayKeyParamValueInfo ();
#endif // _DEBUG

	// we have found the end of a key
	// if there was an informational comment prior to this key
	// it can be associated with the key in the property set
	m_cur_info = MergeMultiInfo ();

	if ( !m_cur_info.empty () )
		m_property_set->SetInfo ( m_cur_key, m_cur_key, m_cur_info );

	else
		m_property_set->SetString ( m_cur_key, m_cur_key, m_cur_info );
	
	// clean up
	m_cur_info = "";

	if ( token == M_LINE_END )
	{
		m_cur_state = PARAM_START;
	}
	else
	if ( token == ' ' || token == '\t' )
	{
		// ignore white space
	}
	else
	{
		m_prev_state = m_cur_state;
		m_cur_state = PARSE_ERROR;
	}
}

void cPropertySetReader::ProcessParam ( const char token )
{
	if ( m_cur_state != PARAM || !m_property_set )
		return;

	if ( token == M_INFO_START )
	{	
		m_prev_state = m_cur_state;
		m_cur_state = PARSE_ERROR;
	}
	else
	if ( token == M_KEY_START )
	{
		m_prev_state = m_cur_state;
		m_cur_state = PARSE_ERROR;
	}
	else
	if ( token == M_KEY_END )
	{
		m_prev_state = m_cur_state;
		m_cur_state = PARSE_ERROR;
	}
	else
	if ( token == M_LINE_END )
	{
		m_prev_state = m_cur_state;
		m_cur_state = PARSE_ERROR;
	}
	else
	if ( token == M_ASSIGN )
	{
		// we have come to the end of a parameter name
		TrimWhiteSpace ( m_cur_param );
		m_cur_state = VALUE;
	}
	else
	if ( token == M_MULTI_VALUE )
	{
		m_prev_state = m_cur_state;
		m_cur_state = PARSE_ERROR;
	}
	else
	{
		// accumulate the parameter
		m_cur_param += token;
	}
}

void cPropertySetReader::ProcessValue ( const char token )
{
	if ( m_cur_state != VALUE || !m_property_set )
		return;

	if ( token == M_MULTI_VALUE )
	{
		TrimWhiteSpace ( m_cur_value );
		m_cur_multi_values.push_back ( m_cur_value );
		m_cur_value = "";
		m_cur_state = MULTI_VALUE;
	}
	else
	if ( token == M_LINE_END )
	{
		TrimWhiteSpace ( m_cur_value );

#ifdef _DEBUG
		cout << "end of a single value..." << endl;
		DisplayKeyParamValueInfo ();
#endif // _DEBUG

		m_property_set->SetString ( m_cur_key, m_cur_param, m_cur_value );

		// process the info for this parameter if there is any
		m_cur_info = MergeMultiInfo ();

		if ( !m_cur_info.empty () )
			m_property_set->SetInfo ( m_cur_key, m_cur_param, m_cur_info );

		// clean up
		m_cur_info = m_cur_param = m_cur_value = "";
		m_cur_state = PARAM_START;
	}
	else
	{
		m_cur_value += token;
	}
}

void cPropertySetReader::ProcessMultiValue ( const char token )
{
	if ( m_cur_state != MULTI_VALUE || !m_property_set )
		return;

	if ( token == M_MULTI_VALUE )
	{
		TrimWhiteSpace ( m_cur_value );
		m_cur_multi_values.push_back ( m_cur_value );
		m_cur_value = "";
	}
	else
	if ( token == M_LINE_END )
	{
		TrimWhiteSpace ( m_cur_value );
		m_cur_multi_values.push_back ( m_cur_value );
		m_cur_value = "";

		// for each multi value
		//		set the value to the appropriate index based on the key and param name
		
		// if info, write the info to the parameter name

		// clean up
		//		reset info, param, and cur value
		//		delete all multi values

#ifdef _DEBUG
		cout << "hit end of multi-value..." << endl;
#endif // _DEBUG

		for ( short i = 0; i < m_cur_multi_values.size (); ++i )
		{

#ifdef _DEBUG			
			cout << "processing multi-value= " << m_cur_multi_values [ i ] << endl;
#endif // _DEBUG

			m_property_set->SetString ( m_cur_key, m_cur_param, m_cur_multi_values [ i ], i );
		}

#ifdef _DEBUG
		DisplayMultiValues ();
#endif // _DEBUG

		// if there is info, process it
		m_cur_info = MergeMultiInfo ();

		if ( !m_cur_info.empty () )
			m_property_set->SetInfo ( m_cur_key, m_cur_param, m_cur_info );

		// clean up
		m_cur_info = m_cur_param = m_cur_value = "";
		m_cur_multi_values.erase ( m_cur_multi_values.begin (), m_cur_multi_values.end () );
		m_cur_state = PARAM_START;
	}
	else
	{
		m_cur_value += token;
	}
}

void cPropertySetReader::ProcessEOF ()
{
	// After we encounter EOF the accepting states are:
	// PARAM_START
	// KEY_END

	// non-accepting states are:
	// KEY
	// INFO
	// VALUE
	// MULTI_VALUE
	// PARAM
	//	PARSE_ERROR
	switch ( m_cur_state )
	{
		// accepting states
		case PARAM_START:
		case KEY_END:		return;

		// non-accepting states
		case KEY:			// for some reason the key is mangled, can't do anything

		case INFO:			// we can't associate the info with a key or param name, oh well...

		case VALUE:			// probably forgot the line end after the param,
							// finish this off
							// should be ok after this
							ProcessValue ( M_LINE_END ); break;

		case MULTI_VALUE:	// forgot line end after the last param
							// finish it off
							ProcessMultiValue ( M_LINE_END ); break;

		case PARAM:			// missing a value
							// FIXME: maybe throw an exception with this message

		case PARSE_ERROR:	;// couldn't recover from an error, oh well...
	}
}

void cPropertySetReader::ProcessError ()
{	
	// based on the last state we were in, try to recover from the error
	
	if ( m_cur_state != PARSE_ERROR )
		return;

	if ( m_prev_state == PARAM_START )
	{
		// we got a key end symbol or an assign symbol
		// skip to the end of the line and reset to param start
		if ( SkipLine () )
		{
			m_prev_state = m_cur_state;
			m_cur_state = PARAM_START;
		}
	}
	else
	if ( m_prev_state == KEY_END )
	{
		// we didn't get a line end or white space
		// skip to the end of the line and reset to param start
		if ( SkipLine () )
		{
			m_prev_state = m_cur_state;
			m_cur_state = PARAM_START;
		}
	}
	else
	if ( m_prev_state == PARAM )
	{
		// we can only get an assign symbol, white or non-white space
		// that means we got a character that can't be part of the param name
		// the param name is currently invalid so delete it
		// if there was info associated with this param name delete it since param is invalid
		// skip to the end of the line and reset to param start
		
		m_cur_info = m_cur_param = "";
		
		if ( SkipLine () )
		{
			m_prev_state = m_cur_state;
			m_cur_state = PARAM_START;
		}
	}

	// if we couldn't recover we are SOL. Just let ProcessEOF() worry about it.
}

bool cPropertySetReader::SkipLine ()
{

	bool no_eof = true;
	char next = '0';

	while ( (next = GetNextChar ()) != M_LINE_END )
	{
		// make sure we didn't hit EOF prematurely
		if ( next == EOF )
		{
			no_eof = false;
			break;
		}
	}

	return no_eof;
}

// utility
void cPropertySetReader::TrimWhiteSpace ( string &toTrim )
{
	// white space is ' ', '\t'
	// trims leading and trailing whitespace
	if ( toTrim.empty () )
		return;
	
	// left side
	while ( !toTrim.empty () && (toTrim.at ( 0 ) == ' ' || toTrim.at ( 0 ) == '\t') ) 
		toTrim.erase ( 0, 1 );
	

	// right side
	short pos = toTrim.length () - 1;
	while ( !toTrim.empty () && (toTrim.at ( pos ) == ' ' || toTrim.at ( pos ) == '\t') ) 
	{
		toTrim.erase ( pos, pos + 1 );
		--pos;
	}
}

string cPropertySetReader::MergeMultiInfo ()
{
	string temp = "";

	// for every string in m_cur_multi_info, concatenate to a single string
	for ( short i = 0; i < m_cur_multi_info.size (); ++i )
		temp += m_cur_multi_info [ i ];

	// clean up
	m_cur_multi_info.erase ( m_cur_multi_info.begin (), m_cur_multi_info.end () );

	return temp;
}

#ifdef _DEBUG
void cPropertySetReader::DisplayKeyParamValueInfo ()
{
	cout << "key=" << m_cur_key << endl;
	cout << "param=" << m_cur_param << endl;
	cout << "value=" << m_cur_value << endl;
	cout << "info=";

	string temp = "";

	// for every string in m_cur_multi_info, concatenate to a single string
	for ( short i = 0; i < m_cur_multi_info.size (); ++i )
		temp += m_cur_multi_info [ i ];
	
	cout << temp << endl << endl;
}

void cPropertySetReader::DisplayMultiValues ()
{
	cout << "key=" << m_cur_key << endl;
	cout << "param=" << m_cur_param << endl;
	cout << "info=";

	string temp = "";

	short i = 0;
	// for every string in m_cur_multi_info, concatenate to a single string
	for ( i = 0; i < m_cur_multi_info.size (); ++i )
		temp += m_cur_multi_info [ i ];
	
	cout << temp << endl;

	cout << "multi-values=";

	for ( i = 0; i < m_cur_multi_values.size (); ++i )
		cout << m_cur_multi_values [ i ] << ',';

	cout << endl;
}
#endif // _DEBUG
