/*********************************************************************************************
*	Copyright (C) 2002 Robert Farrell
*
*	This program is free software; you can redistribute it and/or
*	modify it under the terms of the GNU General Public License
*	as published by the Free Software Foundation; either version 2
*	of the License, or (at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
*
*	See the GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
**********************************************************************************************/
/*********************************************************************************************
*	Class:		cPropertySetReader
*
*	Purpose:	Parses a property set file and stores data in a cPropertySet.
*				
*	Notes:		A simple state machine is used to parse the input file:
*
*				info		param-start	key		key-end		param	value		multi-value
* -------------------------------------------------------------------------------------------
* info-start	info		info		key		error		error	value		multi-value
* key-start		info		key			key		error		error	value		multi-value
* key-end		info		error		key-end	error		error	value		multi-value
* line-end		param-start	param-start	key		param-start	error	param-start	param-start
* assign		info		error		key		error		value	value		multi-value
* non-white		info		param		key		error		param	value		multi-value
* white space	info		param-start	key		key-end		param	value		multi-value
* multi-value	info		error		key		error		error	multi-value	multi-value
*
* ProcessEOF ()
* After we encounter EOF the accepting states are:
* PARAM_START
* KEY_END
*
* non-accepting states are:
* KEY
* INFO
* VALUE
* MULTI_VALUE
* PARAM
* PARSE_ERROR
*
*	This class uses the state machine in the following way:
*	Info is processed by stripping the M_INFO_START delimiter from the string and
*	accumulating the string in the vector m_cur_multi_info. The vector holds single
*	lines of a multi-line comment. Info is associated with a key when a key
*	is processed or for a parameter when a value or multi-value is processed.
*
*	Keys are processed when M_LINE_END is reached and in KEY_END state. Any comment before 
*	the key is stored in the property set as key=keyname, param=keyname.
*	All keys are stored as key=keyname, param=keyname.
*	Key delimiters are stripped from the key.
*
*	Single value parameters are processed when a token is M_LINE_END and 
*	the current state is VALUE. Any info comment is associated with the param name,
*	and the key/value pair is added to the property set.
*
*	Multi-valued parameters are processed when a token is M_LINE_END and
*	the current state is MULTI_VALUE. As each M_MULTI_VALUE token is encountered, the
*	current value is pushed into the vector, when M_LINE_END is reached, each value in
*	the vector is added to the property set. Any info is associated with the param name.
*
*	Attempts to recover from parsing errors are to skip the line in the file that caused
*	the error.
*
*	FIXME:	1. Currently if info is not textually before a key or parameter, info is not saved.
*			3. M_LINE_END in info comments are preserved.
*			4. This class doesn't check for "orphaned" parameters, i.e., all parameters must
*			come after a key, this class just sets up a key of NONE
*			in case params come before the first key in the file.
*			5. Class does not throw an exception for any kind of parse error, should it? Can't
*			really do anything about it if a parse error happens, one less exception to catch...
*			6. Create a state transition graph and include in documentation.
*			7. Doesn't handle tabs in comments yet.
*
*	Author:	Robert Farrell
*	Email:	farrellr@accis.edu
*
*	History:
*	12/18/2001	Initial version.
*	1/8/2002	Properly strips key-start, key-end, and info-start delimiters.
*				Keys with no parameters are now preserved in the file.
*********************************************************************************************/
#ifndef _PROPERTY_SET_READER_HPP
#define _PROPERTY_SET_READER_HPP

#include "ps_property_set.hpp"
#include <fstream>

using std::fstream;
using std::string;

class cPropertySetReader
{
public:
	cPropertySetReader ( cPropertySet &ps );
	virtual ~cPropertySetReader ();

	void Read ();

private:
	enum eParseState {	INFO,
						PARAM_START,
						PARAM,
						KEY,
						KEY_END,
						VALUE,
						MULTI_VALUE,
						PARSE_ERROR };

	char GetNextChar ();
	void ProcessInfo ( const char token );
	void ProcessParamStart ( const char token );
	void ProcessKey ( const char token );
	void ProcessKeyEnd ( const char token );
	void ProcessParam ( const char token );
	void ProcessValue ( const char token );
	void ProcessMultiValue ( const char token );
	void ProcessEOF ();
	void ProcessError ();

	// returns false if hit EOF before skipping line
	bool SkipLine ();

	// utility
	void TrimWhiteSpace ( string &toTrim );

	// combines all multi-line info comments into a single string
	string	MergeMultiInfo ();

#ifdef _DEBUG
	void DisplayKeyParamValueInfo ();
	void DisplayMultiValues ();
#endif // _DEBUG

	// no copying allowed
	cPropertySetReader ( const cPropertySetReader & );
	cPropertySetReader & operator= ( const cPropertySetReader & );

private:
	cPropertySet *		m_property_set;
	fstream				m_in;
	eParseState			m_cur_state;
	eParseState			m_prev_state;	// when a PARSE_ERROR occurs, this is set, otherwise ignored
	string				m_cur_key;
	string				m_cur_param;
	string				m_cur_value;
	string				m_cur_info;
	std::vector<string>	m_cur_multi_values;
	std::vector<string>	m_cur_multi_info;
	const char			M_INFO_START;
	const char			M_KEY_START;
	const char			M_KEY_END;
	const char			M_LINE_END;
	const char			M_ASSIGN;
	const char			M_MULTI_VALUE;
};
#endif // _PROPERTY_SET_READER_HPP
