/*********************************************************************************************
*	Copyright (C) 2002 Robert Farrell
*
*	This program is free software; you can redistribute it and/or
*	modify it under the terms of the GNU General Public License
*	as published by the Free Software Foundation; either version 2
*	of the License, or (at your option) any later version.
*
*	This program is distributed in the hope that it will be useful,
*	but WITHOUT ANY WARRANTY; without even the implied warranty of
*	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
*
*	See the GNU General Public License for more details.
*
*	You should have received a copy of the GNU General Public License
*	along with this program; if not, write to the Free Software
*	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
**********************************************************************************************/
#include "ps_writer.hpp"

#ifdef _DEBUG
#include <iostream>
using std::cout;
using std::endl;
#endif // _DEBUG

using std::ios;

cPropertySetWriter::cPropertySetWriter ( cPropertySet &ps ) :
m_property_set ( &ps ),
M_INFO_START ( ps.config.GetInfoStartDelim () ),
M_KEY_START ( ps.config.GetKeyStartDelim () ),
M_KEY_END ( ps.config.GetKeyEndDelim () ),
M_LINE_END ( ps.config.GetLineEndDelim () ),
M_ASSIGN ( ps.config.GetAssignSymbol () ),
M_MULTI_VALUE ( ps.config.GetMultiValueDelim () )
{
	stringstream buffer;
	string file = ps.config.GetFileName ();

	// open the file based on the constraints
	if ( ps.config.GetFileMode () == cPropertySetConfig::EXISTS_OVERWRITE_OR_CREATE )
	{	
		// overwrite file in either case
		m_out.open ( file.c_str (), ios::out );
		if ( !m_out )
		{
			buffer << "Error: File IO error while attempting to open file: " << file.c_str ();
			cPropertySet::cFileModeConstraint e ( buffer.str () );
			throw e;
		}
	}
	else
	if ( ps.config.GetFileMode () == cPropertySetConfig::EXISTS_OVERWRITE_OR_FAIL )
	{
		m_out.open ( file.c_str (), ios::in );
		if ( !m_out )
		{
			// file doesn't exist so fail
			buffer << "Error: File IO error while attempting to open existing file: " << file.c_str ();
			cPropertySet::cFileModeConstraint e ( buffer.str () );
			throw e;
		}
		else
		{
			// file exists, so overwrite
			m_out.close ();
			m_out.clear ();
			m_out.open ( file.c_str (), ios::out );
			if ( !m_out )
			{
				// file io error
				buffer << "Error: File IO error while attempting to open existing file: " << file.c_str ();
				cPropertySet::cFileModeConstraint e ( buffer.str () );
				throw e;
			}
		}
	}
	else
	if ( ps.config.GetFileMode () == cPropertySetConfig::EXISTS_FAIL_OR_CREATE )
	{
		// see if the file exists
		m_out.open ( file.c_str (), ios::in );
		if ( !m_out )
		{
			// file doesn't exist so create it
			m_out.clear ();
			m_out.open ( file.c_str (), ios::out );
			if ( !m_out )
			{
				buffer << "Error: File IO error while attempting to create file: " << file.c_str ();
				cPropertySet::cFileModeConstraint e ( buffer.str () );
				throw e;
			}
		}
		else
		{
			// file exists, so fail
			buffer << "Error: File: " << file.c_str () << " exists and should not";
			cPropertySet::cFileModeConstraint e ( buffer.str () );
			throw e;
		}
	}
	else
	{
		buffer << "Error: File constraint not specified for file: " << file.c_str ();
		cPropertySet::cFileModeConstraint e ( buffer.str () );
		throw e;
	}
}

cPropertySetWriter::~cPropertySetWriter ()
{
	m_out.clear ();
	m_out.close ();
}

void cPropertySetWriter::Write ()
{
	// iterate through all the keys and all the parameter values
	// if there is info associated with a key, write that first
	// then write the key once
	// after the key, write all the parameter names and values
	// if a parameter has info associated with it, write it before the parameter

	// example:
	// M_INFO_START info
	// M_KEY_START key M_KEY_END<M_LINE_END>
	// M_INFO_START info
	// parameter<M_ASSIGN>value <M_MULTI_VALUE> ... <M_LINE_END>

	if ( !m_out )
		return;

	std::multimap < string, cMultiProperty >::const_iterator cur_key;
	std::multimap < string, cMultiProperty >::const_iterator cur_param;

	for ( (cur_key = m_property_set->m_property_set.begin () ); cur_key != m_property_set->m_property_set.end (); cur_key = cur_param )
	{
		WriteInfo ( m_property_set->GetInfo ( cur_key->first, cur_key->first ) );
		WriteKey ( cur_key->first );

		// try to find the parameters
		for ( cur_param = m_property_set->m_property_set.lower_bound ( cur_key->first ); cur_param != m_property_set->m_property_set.upper_bound ( cur_key->first ); ++cur_param )
		{
			// don't write the parameter key=key name, param=key name
			if ( cur_param->second.Name () != cur_key->first )
			{
				WriteInfo ( m_property_set->GetInfo ( cur_key->first, cur_param->second.Name () ) );
				WriteParam ( cur_param->second.Name () );
				WriteMultiProperty ( const_cast< cMultiProperty& >( cur_param->second ) );
			
				m_out << M_LINE_END;
			} // end if
		} // end for

		m_out << M_LINE_END;
	} // end for
}

void cPropertySetWriter::WriteKey ( const string &key )
{
	if ( !m_out )
		return;

#ifdef _DEBUG	
	cout << "writing key=" << key.c_str () << endl;
#endif // _DEBUG

	if ( !key.empty () )
		m_out << M_KEY_START << ' ' << key.c_str () << ' ' << M_KEY_END << M_LINE_END;

#ifdef _DEBUG
	m_out.flush ();
#endif // _DEBUG
}

void cPropertySetWriter::WriteInfo ( const string &info )
{
	if ( !m_out )
		return;

	if ( !info.empty () )
	{

#ifdef _DEBUG
		cout << "writing info=" << info.c_str () << endl;
#endif // _DEBUG
		
		m_out << M_LINE_END << M_INFO_START << ' ';

		// for every M_LINE_END, start a new M_INFO_START
		for ( int i = 0; i < info.length (); ++i )
		{
			m_out << info.at ( i );

			if ( (info.at ( i ) == M_LINE_END) && (i < info.length () - 1) )	
				m_out << M_INFO_START << ' ';
		}
	}

#ifdef _DEBUG
	m_out.flush ();
#endif // _DEBUG
}

void cPropertySetWriter::WriteParam ( const string &name )
{
	if ( !m_out )
		return;

#ifdef _DEBUG
	cout << "writing param=" << name.c_str () << endl;
#endif // _DEBUG

	if ( !name.empty () )
		m_out << name.c_str () << M_ASSIGN;

#ifdef _DEBUG
	m_out.flush ();
#endif // _DEBUG
}

void cPropertySetWriter::WriteMultiProperty ( cMultiProperty &p )
{
	if ( !m_out )
		return;

	// if a multi property has more than one value, use M_MULTI_VALUE to separate the values
	for ( unsigned int i = 0; i < p.NumValues (); ++i )
	{

#ifdef _DEBUG
		cout << "writing multi-property=" << p.GetString ( i ).c_str () << endl;
#endif // _DEBUG

		m_out << p.GetString ( i );

		if ( i < p.NumValues () - 1 )
			m_out << M_MULTI_VALUE;
	}
#ifdef _DEBUG
	m_out.flush ();
#endif // _DEBUG
}

